# R code: Example 3.3
# File: ctarch.eigen.r 
#
# Functions to compute and plot eigenvalues, eigenfunctions
# writen by Daren B.H. Cline, Department of Statistics, Texas A&M University
# last modified Nov 2006.
#
# Minor modification: call to "atan" replaced by call to "atan2" (i.e. 2 arguments).
####################################################################################

ctarch.eigen = function(arpars=c(a11,a12,-0.4,-0.1),chpars=c(0.7,0.2,0.3,0.1),rx=c(0,1,2), 
   disc.pts=numeric(0), find.dist=TRUE, nefpts=200,quad.del=.1,max.iter=100, 
   err=1e-7, save.file=NA, verbose=FALSE, ...){
# rx = 0 ==> find the Lyapounov exponent gamma for TARCH(2)
# rx = r ==> solve the eigenvalue and eigenfunction problem for CTARCH(2) 
#            that determines existence of r-moment for TARCH(2)
# single threshold at cos(arc(theta)) = 0 is assumed for now (delay = 1)
# error distribution is assumed normal(0,1)
# arpars = (a11, a12, a21, a22);  chpars = (b11, b12, b21, b22)

# OUTPUT are gamma and equilbrium solution nu
# also output are right e-function lambda_r and left e-function pi_r on [-pi,pi]
# and the eigenvalue (with error flag if not found to be equal for left and right)

# nefpts = number of points of the eigenfunctions per quadrant
# note evaluations at -pi, -pi/2, pi/2, pi are excluded
# quadrature uses 10/quad.del evenly spaced points in (-5,5), offcentered
###################################################################################

pt1        = proc.time()          # for timing the execution of this function
rx         = unique(sort(rx))
find.gamma = (any(rx==0))

if (find.gamma) rxp = rx[rx!=0]
else rxp = rx
lrx      = length(rxp)
sdj      = length(rx)+find.dist   # index for stationary distribution
err.flag = rep(FALSE,sdj)
iter     = rep(0,sdj)

# set up quadrature
nqpts   = ceiling(5/quad.del)
quadpts = quad.del*((1-nqpts):nqpts)-.5*quad.del
quadwts = dnorm(quadpts)
quadwts = quadwts/sum(quadwts)

# set up interpolation knots
def   = .5*pi/nefpts
arcth = def*((1-2*nefpts):(2*nefpts))-.5*def
cosa  = cos(arcth)
sina  = sin(arcth)
cota1 = (1/tan(c(arcth-.5*def,pi)))[1:(2*nefpts+1)]
cota1[c(1,nefpts+1,2*nefpts+1)] = c(Inf,0,-Inf)
cpos  = (cosa>0)
spos  = (sina>0)

# compute AR and CH for the knots, also ZZ, Wsq and logW
rega = ifelse(cosa<0,1,3)       # threshold at cos = 0 (delay = 1)
AR   = arpars[rega]*cosa + arpars[rega+1]*sina
CH   = sqrt((chpars[rega]*cosa)^2 + (chpars[rega+1]*sina)^2)
ZZ   = AR+CH%*%t(quadpts)       # 4*nefpts x 2*nqpts, same below
alphas = atan2(cosa,ZZ)         # angles of "new" thetas
Wsq  = ZZ^2 + cosa^2
logW = .5*log(Wsq)

# compute transition matrices DFn and DFp
DFn = pnorm((abs(cosa[!cpos])%*%t(cota1)-AR[!cpos])/CH[!cpos])
DFn = DFn[,1:(2*nefpts)]-DFn[,2:(2*nefpts+1)]
DFp = pnorm((cosa[cpos]%*%t(cota1)-AR[cpos])/CH[cpos])
DFp = DFp[,1:(2*nefpts)]-DFp[,2:(2*nefpts+1)]

rm("AR","CH","ZZ","cosa","sina","cota1") # remove unneeded large matrices

# set up output matrices
nu     = rep(0,4*nefpts)
gam    = Inf
Pi     = rep(.5/pi,4*nefpts)
Pi.new = Pi
Lambda = matrix(1,lrx,4*nefpts)
Rho    = rep(Inf,lrx)

if(find.gamma){    # solve for gamma, nu constrained to sum to 0
# initialize
nu.diff = 1
nu.iter = 0

# iterate
while((nu.diff>err)&(nu.iter<max.iter)){
   nu.iter = nu.iter + 1
   nu.old  = nu
   # interpolate nu at values in alphas, (assuming nu is continuous on each quadrant)
   nu.at.alphas = lin.interp(arcth,nu.old,alphas,disc.pts,expedite=T)$y  
   nu.new  = (nu.at.alphas+logW)%*%quadwts      # this is the quadrature
   gam.new = max(nu.new-nu.old)
   nu      = nu.new - mean(nu.new)
   nu.diff = max(abs(nu-nu.old))

if(verbose) cat("iter =",nu.iter,"gamma = ",gam.new,"diff =",nu.diff,"\n")

} # end "while"
gam     = gam.new
nu      = as.vector(nu)
iter[1] = nu.iter

# estimate quadrature error
integrands  = t(nu.at.alphas)*quadwts   # columns are the integrands for a fixed theta
second.diff = diff(integrands,differences=2)
quad.err    = .25*quad.del*max(colSums(abs(second.diff)))

if(nu.diff>err) err.flag[1] = TRUE

} # end "if"

if(find.dist){   # find stationary distribution
# initialize
pi.diff = 1
pi.iter = 0

# iterate
while((pi.diff>err)&(pi.iter<max.iter)){
   pi.iter       = pi.iter + 1
   Pi.old        = Pi
   Pi.new[!cpos] = Pi.old[!spos]%*%DFn
   Pi.new[cpos]  = Pi.old[spos]%*%DFp
   Pi            = Pi.new/(def*sum(Pi.new))
   pi.diff       = max(abs(Pi-Pi.old))

if(verbose) cat("iter =",pi.iter,"diff =",pi.diff,"\n")

} # end "while"

Pi        = as.vector(Pi)
iter[sdj] = pi.iter

if(pi.diff>err) err.flag[sdj] = TRUE

} # end if

if(lrx>0){   # iterate over the nonzero exponents in rx
for (rj in 1:lrx){
   rr = rxp[rj]
   Wr = Wsq^(rr/2)

# integrating over theta1 is easier but will not give the stationary distribution

# initialize
lam      = exp(rr*nu)
lam.diff = 1
lam.iter = 0

# iterate
while((lam.diff>err)&(lam.iter<max.iter)){
   lam.iter = lam.iter + 1
   lam.old  = lam
   # interpolate lam at values in alphas, (assuming lam is continuous on each quadrant)
   lam.at.alphas = lin.interp(arcth,lam.old,alphas,disc.pts,expedite=T)$y  
   lam.new       = (lam.at.alphas*Wr)%*%quadwts      # this is the quadrature
   rho.new       = max(lam.new/lam.old)
   lam           = lam.new/mean(lam.new)
   lam.diff      = max(abs(lam-lam.old))

if(verbose) cat("iter =",lam.iter,"rho = ",rho.new,"diff =",lam.diff,"\n")

} # end "while"
iter[rj+find.gamma] = lam.iter
Rho[rj]             = rho.new
Lambda[rj,]         = as.vector(lam)

if(lam.diff>err) err.flag[rj+find.gamma] = TRUE

} # end "for"
} # end "if"

pt2          = proc.time()
cpu.time     = signif(pt2[1]-pt1[1],2)
elapsed.time = signif(pt2[3]-pt1[3],2)

if(verbose) cat("err.flag =",err.flag,"  cpu.time =",cpu.time,"\n")

result = list(gamma=gam,arcth=arcth,nu=nu,Pi=Pi,Rho=Rho,Lambda=Lambda,quad.err=quad.err,
   disc.pts=disc.pts,iter=iter,err.flag=err.flag,arpars=arpars,chpars=chpars,rx=rx,
   nefpts=nefpts,quad.del=quad.del,quadpts=quadpts,max.iter=max.iter,err=err,
   program="ctarch.eigen",date=as.character(Sys.time()),cpu.time=cpu.time,
   elapsed.time=elapsed.time,save.file=save.file)
if(!is.na(save.file)) invisible(dput(result,file=save.file,control="all"))
return(result)
} # end "function"


lin.interp = function(x, y=NULL, xnew, disc.pts=numeric(0), remove.na=TRUE, 
   expedite=FALSE){
# linear interpolation of a piecewise continuous function
# with linear extension beyond the knots near a discontinuity or at the extremes
# y = function values at the points in x
# xnew = points at which the function is to be interpolated
# disc.pts = points of discontinuity of the function
# pts in both x and disc.pts are assumed to be continuity pts with value in y
# a point in xnew that is very close to one in disc.pts may be evaluated 
# inappropriately if there is round off error - no warning is given

# output x is xnew
# output y has the same attributes as xnew
# if remove.na = TRUE then points that cannot be interpolated are removed
# expedite = T assumes valid, uniques, sorted vectors; and that x and 
# disc.pts have no common values

if (!is.null(y)) {
   xy = input.xy(x,y)
   x  = xy$x
   y  = xy$y
} # end "if"
new.attr = attributes(xnew)
new.l    = length(xnew)

if (expedite){   # assumes user inputs valid data (see below)
ldc = length(disc.pts)
nx  = length(x)
xx  = x
yy  = y
} # end "if"

else{   # validate the inputs
xx = unique(x[is.numeric(x)])
nx = length(x)
if (length(xx)<nx) stop("input x does not have unique numerical elements\n")
if (length(y)!=nx) stop("input x and y do not have the same length\n")
if (!all(is.numeric(y))) stop("input y has non-numerical elements\n")
if (!all(is.numeric(xnew))) stop("input xnew has non-numerical elements\n")
if ((!is.null(disc.pts))&(!all(is.numeric(disc.pts))))
   stop("input disc.pts has non-numerical elements\n")

#  and organize x-values and discontinuity points
if (length(disc.pts)>0){
   disc.pts = sort(unique(as.vector(disc.pts)))
   disc.pts = disc.pts[!close.match(disc.pts,x)]   # drop any discontinuity points in x
} # end "if"
ldc = length(disc.pts)
sx  = sort.list(x)
xx  = as.vector(x)[sx]
yy  = as.vector(y)[sx]
} # end "else"

if (ldc==0){   # no need to worry about discontinuities
   slopes   = diff(yy)/diff(xx)
   xnew.int = findInterval(xnew,xx,all.inside=T)
   ynew     = yy[xnew.int] + slopes[xnew.int]*(xnew-xx[xnew.int])
} # end "if"

else{   # worry about discontinuities

# identify which intervals the points are in
idx = findInterval(xx,disc.pts)
idn = findInterval(xnew,disc.pts)

# interpolate within each interval
ynew = rep(NA,length(xnew))
for (j in 0:ldc){
   idnj = (idn==j)
   xnj  = xnew[idnj]
   lnj  = length(xnj)
   if (lnj>0){         # consider only the intervals we need
      idxj = (idx==j)
      xxj  = xx[idxj]
      lxj  = length(xxj)
      if (lxj==0) ynew[idnj] = NA          # cannot interpolate here (no data)
      else if (lxj==1) ynew[idnj] = yy[idxj]   # set constant (only one datum)
      else{
         yyj        = yy[idxj]
         slopes     = diff(yyj)/diff(xxj)
         xnew.int   = findInterval(xnj,xxj,all.inside=T)
         ynew[idnj] = yyj[xnew.int] + slopes[xnew.int]*(xnj-xxj[xnew.int])
      }
   } # end "if"
} # end "for"

} # end "else"

if (remove.na){   # remove the points not interpolated
   xnew = xnew[!is.na(ynew)]
   ynew = ynew[!is.na(ynew)]
} # end "if"

if (length(xnew)==new.l){  # retain attributes of the input
   attributes(xnew) = new.attr
   attributes(ynew) = new.attr
} # end "if"

return(list(x=xnew,y=ynew))
} # end "function"

input.xy = function(x, y=NULL, xname="x", yname="y", extend=FALSE){
# check if x and y are provided as a list or as a matrix or as vectors
# if y = NULL then x must be a list or a matrix with at least two columns
# unless x and y are named, the first two columns or list elements are used
# extend = T: one vector will be replicated to make them equal in length
#    in this case, the shorter inherits the attributes of the longer
# result is a list with elements x and y

# essentially the same as xy.coords???

# select the elements for x and y
if (!is.null(y)){
   xx = x
   yy = y
} # end if
else if (is.list(x)){
   if (xname %in% names(x)) assign("xx",x[[xname]],inherits=T)
   else  assign("xx",x[[1]],inherits=T)
   if (yname %in% names(x)) assign("yy",x[[yname]],inherits=T)
   else assign("yy",x[[2]],inherits=T)
} # end else
else if (is.matrix(x)){
   if (xname %in% colnames(x)) assign("xx",x[,xname],inherits=T)
   else assign("xx",x[,1],inherits=T)
   if (yname %in% colnames(x)) assign("yy",x[,yname],inherits=T)
   else assign("yy",x[,2],inherits=T)
} # end else
else stop("y is missing and x is not a matrix or list\n")

# extend shorter element if desired
if ((extend=T)&(length(xx)!=length(yy))){
   if (length(yy)<length(xx)){
      yy             = rep(yy,length.out=length(xx))
      attributes(yy) = attributes(xx)
   } # end if
   else{
      xx             = rep(xx,length.out=length(yy))
      attributes(xx) = attributes(yy)
   } # end else
} # end if

return(list(x=xx,y=yy))
} # end function

